//*************************************************************************************************
//
//	Description:
//		road_multilayer.fx - A multilayered road surface shader.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     05/05/2007  0.01					Created
//		TNettleship     06/01/2007  0.02					Added optional param for L1 normals.
//		TNettleship			07/11/2007	0.03					Changed lighting to work in world coords.
//		TNettleship			07/24/2007	0.04					Added anisotropic filtering to diffuse maps.
//		TNettleship     08/18/2007  0.05					Added more specialisation.
//		TNettleship			10/23/2007	0.06					Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#define USE_LAYER_3
#define USE_LAYER_4
#define USE_LAYER_5
#define USE_NORMAL_SMOOTHING
#endif

#if defined( _PS3_ ) || defined( _XBOX )
#define DISABLE_NORMAL_MAPPING
#endif



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool export = false;
	bool appEdit = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
> = 0;

int texcoord3 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 3;
	int MapChannel = 3;
	int RuntimeTexcoord = 2;
	bool export = false;
> = 0;

int texcoord4 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 4;
	int MapChannel = 4;
	int RuntimeTexcoord = 3;
	bool export = false;
> = 0;

int texcoord5 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 5;
	int MapChannel = 5;
	int RuntimeTexcoord = 4;
	bool export = false;
> = 0;

#endif

//
// Textures
//

//
// Layer 1 - base diffuse + optional broad normals
//

#ifdef _3DSMAX_
texture layer1DiffuseTexture : DiffuseMap			// Diffuse colour in RGB
#else
texture layer1DiffuseTexture : TEXTURE				// Diffuse colour in RGB
#endif
<
	string UIName = "L1 Diff {UV1}";
	bool appEdit = true;
	bool export = true;
>;


//
// Layer 2 - full layer with DSN maps, diffuse multiplied with L1 always
//

texture layer2DiffuseTexture : TEXTURE				// Diffuse colour in RGB, opacity in Alpha
<
	string UIName = "L2 Diff {UV2}";
	bool appEdit = true;
	bool export = true;
>;

texture layer2SpecularTexture : TEXTURE				// Specular colour in RGB, shininess in alpha
<
	string UIName = "L2 Spec {UV2}";
	bool appEdit = true;
	bool export = true;
>;

texture layer2NormalTexture : TEXTURE					// Normal map
<
	string UIName = "L2 Norm {UV2}";
	bool appEdit = true;
	bool export = true;
>;

float layer2MinSpecPower
<
	string UIName = "Min Spec Power L2";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
	bool export = true;
> = 1.0f;

float layer2MaxSpecPower
<
	string UIName = "Max Spec Power L2";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
	bool export = true;
> = 32.0f;

float layer2GlobalSpecularFactor
<
	string UIName = "Specular Factor L2";
	float UIMin = -10.0f;
	float UIMax = 10.0f;
	bool appEdit = true;
	bool export = true;
> = 1.0f;

//
// Layer 3 - alphaed colour map + specular controls
//

SPECIALISATION_PARAM( useLayer3, "Use Layer 3?", "USE_LAYER_3" )	// TRUE if layer 3 should be used
DECLARE_DEPENDENT_VERTEX_STREAM( texCoord2Dependency, texCoord2, TEXCOORD3, useLayer3 )

#if defined( _3DSMAX_ ) || defined( USE_LAYER_3 )
DEPENDENT_TEXTURE_PARAM( layer3DiffuseTexture, "L3 Diff {UV3}", useLayer3 )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( layer3GlobalSpecularFactor, "Specular Factor L3", useLayer3, -10.0f, 10.0f, 1.0f )
#endif


//
// Layer 4 - alphaed colour map + specular controls + optional smoothing control
//

SPECIALISATION_PARAM( useLayer4, "Use Layer 4?", "USE_LAYER_4" )	// TRUE if layer 4 should be used
DECLARE_DEPENDENT_VERTEX_STREAM( texCoord3Dependency, texCoord3, TEXCOORD4, useLayer4 )

#if defined( _3DSMAX_ ) || defined( USE_LAYER_4 )
DEPENDENT_TEXTURE_PARAM( layer4DiffuseTexture, "L4 Diff {UV4}", useLayer4 )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( layer4GlobalSpecularFactor, "Specular Factor L4", useLayer4, -10.0f, 10.0f, 1.0f )
#endif

SPECIALISATION_PARAM( useNormalSmoothing, "Norm Smoothing?", "USE_NORMAL_SMOOTHING" )	// TRUE if normal smoothing should be used

#if defined( _3DSMAX_ ) || defined( USE_NORMAL_SMOOTHING )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( layer4NormalSmoothingFactor, "Normal Smoothness L4", useNormalSmoothing, -10.0f, 10.0f, 0.0f )
#endif


//
// Layer 5 - colour detail map, mult or added to L1&L2 before lighting
//

SPECIALISATION_PARAM( useLayer5, "USe Layer 5?", "USE_LAYER_5" )	// TRUE if layer 5 should be used
DECLARE_DEPENDENT_VERTEX_STREAM( texCoord4Dependency, texCoord4, TEXCOORD5, useLayer5 )

#if defined( _3DSMAX_ ) || defined( USE_LAYER_5 )
DEPENDENT_BOOL_PARAM( layer5Additive, "L5 Additive?", useLayer5 )
DEPENDENT_TEXTURE_PARAM( layer5DiffuseTexture, "L5 Diff {UV5}", useLayer5 )
#endif


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D layer1DiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="layer1DiffuseTexture"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
> 
= sampler_state
{
	Texture = < layer1DiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};

sampler2D layer2DiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="layer2DiffuseTexture"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
> 
= sampler_state
{
	Texture = < layer2DiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
	#endif
};

sampler2D layer2SpecularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="layer2SpecularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
>
= sampler_state
{
	Texture = < layer2SpecularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
	#endif
};

sampler2D layer2NormalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="layer2NormalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < layer2NormalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( USE_LAYER_3 )
sampler2D layer3DiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="layer3DiffuseTexture"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
>
= sampler_state
{
	Texture = < layer3DiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( USE_LAYER_4 )
sampler2D layer4DiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="layer4DiffuseTexture"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
>
= sampler_state
{
	Texture = < layer4DiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( USE_LAYER_5 )
sampler2D layer5DiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="layer5DiffuseTexture"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
> 
= sampler_state
{
	Texture = < layer5DiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};
#endif



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
#ifdef _3DSMAX_
	float3 position		: POSITION;													// Object space position
	float4 colour			: TEXCOORD1;												// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV channel 1 texture coord
	float2 texCoord1	: TEXCOORD2;												// UV channel 2 texture coord
	float2 texCoord2	: TEXCOORD3;												// UV channel 3 texture coord
	float2 texCoord3	: TEXCOORD4;												// UV channel 4 texture coord
	float2 texCoord4	: TEXCOORD5;												// UV channel 5 texture coord
	float3 normal			: NORMAL;														// Object space normal
	float3 tangent		: TANGENT;													// Object space tangent
	float3 binormal		: BINORMAL;													// Object space binormal
#else
	float3 position		: POSITION;													// Object space position
	float4 colour			: COLOR;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV channel 1 texture coord
	float2 texCoord1	: TEXCOORD1;												// UV channel 2 texture coord
#if defined( USE_LAYER_3 )
	// This vertex stream isn't needed without layer 3
	float2 texCoord2	: TEXCOORD2;												// UV channel 3 texture coord
#endif
#if defined( USE_LAYER_4 )
	// This vertex stream isn't needed without layer 4
	float2 texCoord3	: TEXCOORD3;												// UV channel 4 texture coord
#endif
#if defined( USE_LAYER_5 )
	// This vertex stream isn't needed without layer 5
	float2 texCoord4	: TEXCOORD4;												// UV channel 5 texture coord
#endif
	float3 normal			: NORMAL;														// Object space normal
	float3 tangent		: TANGENT;													// Object space tangent
	float3 binormal		: BINORMAL;													// Object space binormal
#endif
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD6;														// Vertex colour
	float4 colour2		: TEXCOORD7;														// Vertex colour
	float4 texCoord01	: TEXCOORD0;												// UV coords for texture channels 0 & 1
	float4 eye				: TEXCOORD1;												// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( USE_LAYER_3 ) || defined( USE_LAYER_4 )
	// This vertex stream isn't needed without layer 3 or layer 4
	float4 texCoord23	: TEXCOORD2;												// UV coords for texture channels 2 & 3
#endif

#if defined( _3DSMAX_ ) || defined( USE_LAYER_5 )
	// This vertex stream isn't needed without layer 5
	float4 texCoord4_and_binormal_xy	: TEXCOORD3;			// UV coords for texture channel 4, with binormal vector xy in the zw components
	float4 normal_and_binormal_z			: TEXCOORD4;			// Normal vector (world space), with binormal vector z in the w component
	float3 tangent										: TEXCOORD5;			// Tangent vector (world space)
#else
	float3 normal		: TEXCOORD3;												// Normal vector (world space)
	float3 tangent	: TEXCOORD4;												// Tangent vector (world space)
	float3 binormal : TEXCOORD5;												// Binormal vector (world space)
#endif

	DECLARE_LIGHTING_INTERPOLATORS_VS( 8 )
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT RoadVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.colour		= float4( 1.0f, 1.0f, 1.0f, 1.0f );
	_output.colour2		= _input.colour;
	_output.texCoord01.xy = _input.texCoord0;
	_output.texCoord01.zw = _input.texCoord1;
#if defined( _3DSMAX_ ) || defined( USE_LAYER_3 ) || defined( USE_LAYER_4 )
		_output.texCoord23 = float4( 0.0f, 0.0f, 0.0f, 0.0f );
#endif

	DEPENDENT_CODE_START( useLayer3 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_3 )
		_output.texCoord23.xy = _input.texCoord2;
#endif
	DEPENDENT_CODE_END( useLayer3 )

	DEPENDENT_CODE_START( useLayer4 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_4 )
		_output.texCoord23.zw = _input.texCoord3;
#endif
	DEPENDENT_CODE_END( useLayer4 )

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	float3 binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;

	// Pack into output params (differently depending on if layer 5 is present)
	DEPENDENT_CODE_START( useLayer5 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_5 )
		_output.texCoord4_and_binormal_xy = float4( _input.texCoord4, binormal.xy );
		_output.normal_and_binormal_z			= float4( normal, binormal.z );
		_output.tangent										= mul( float4( _input.tangent, 0.0f ), world ).xyz;
#endif
	DEPENDENT_CODE_ELSE( useLayer5 )
#if defined( _3DSMAX_ ) || !defined( USE_LAYER_5 )

#if defined( _3DSMAX_ )
		_output.texCoord4_and_binormal_xy = float4( 0.0f, 0.0f, binormal.xy );
		_output.normal_and_binormal_z			= float4( normal, binormal.z );
		_output.tangent										= mul( float4( _input.tangent, 0.0f ), world ).xyz;
#else
		_output.normal   = normal;
		_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
		_output.binormal = binormal;
#endif

#endif
	DEPENDENT_CODE_END( useLayer5 )

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	float globalSpecularFactorValue = layer2GlobalSpecularFactor;
	float minSpecPowerValue = layer2MinSpecPower;
	float maxSpecPowerValue = layer2MaxSpecPower;

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour											: TEXCOORD6;					// Vertex colour
	float4 colour2										: TEXCOORD7;					// Vertex colour
	float4 texCoord01									: TEXCOORD0;			// UV coords for texture channels 0 & 1
	float4 eye												: TEXCOORD1;			// Eye vector (world space)
	float4 texCoord23									: TEXCOORD2;			// UV coords for texture channels 2 & 3
	float4 texCoord4_and_binormal_xy	: TEXCOORD3;			// UV coords for texture channel 4, with binormal vector xy in the zw components
	float4 normal_and_binormal_z			: TEXCOORD4;			// Normal vector (world space), with binormal vector z in the w component
	float3 tangent										: TEXCOORD5;			// Tangent vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 8 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD6_centroid;							// Vertex colour
	float4 colour2		: TEXCOORD7_centroid;							// Vertex colour
	float4 texCoord01	: TEXCOORD0;												// UV coords for texture channels 0 & 1
	float4 eye				: TEXCOORD1_centroid;								// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( USE_LAYER_3 ) || defined( USE_LAYER_4 )
	// This vertex stream isn't needed without layer 3 or layer 4
	float4 texCoord23	: TEXCOORD2;												// UV coords for texture channels 2 & 3
#endif

#if defined( _3DSMAX_ ) || defined( USE_LAYER_5 )
	// This vertex stream isn't needed without layer 5
	float4 texCoord4_and_binormal_xy	: TEXCOORD3;			// UV coords for texture channel 4, with binormal vector xy in the zw components
	float4 normal_and_binormal_z			: TEXCOORD4;			// Normal vector (world space), with binormal vector z in the w component
	float3 tangent										: TEXCOORD5;			// Tangent vector (world space)
#else
	float3 normal		: TEXCOORD3;												// Normal vector (world space)
	float3 tangent	: TEXCOORD4;												// Tangent vector (world space)
	float3 binormal : TEXCOORD5;												// Binormal vector (world space)
#endif

	DECLARE_LIGHTING_INTERPOLATORS_PS( 8 )
	DECLARE_SHADOW_PS_INPUTS
};

#endif


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

//REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT RoadFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read textures
	float4 layer1DiffuseTexColour = tex2D( layer1DiffuseMap, _input.texCoord01.xy );
	float4 layer2DiffuseTexColour = tex2D( layer2DiffuseMap, _input.texCoord01.zw );
	float4 layer2SpecularTexColour = tex2D( layer2SpecularMap, _input.texCoord01.zw );
#if !defined( DISABLE_NORMAL_MAPPING )
	float4 layer2NormalMapColour = tex2D( layer2NormalMap, _input.texCoord01.zw );
#endif
	float4 layer3TexColour, layer4TexColour, layer5TexColour;

	DEPENDENT_CODE_START( useLayer3 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_3 )
		layer3TexColour = tex2D( layer3DiffuseMap, _input.texCoord23.xy );
#endif
	DEPENDENT_CODE_END( useLayer3 )

	DEPENDENT_CODE_START( useLayer4 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_4 )
		layer4TexColour = tex2D( layer4DiffuseMap, _input.texCoord23.zw );
#endif
	DEPENDENT_CODE_END( useLayer4 )

	DEPENDENT_CODE_START( useLayer5 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_5 )
		layer5TexColour = tex2D( layer5DiffuseMap, _input.texCoord4_and_binormal_xy.xy );
#endif
	DEPENDENT_CODE_END( useLayer5 )

#if defined( DISABLE_NORMAL_MAPPING )
	float3 TSnormal;
	float3 eye = normalize( _input.eye.xyz );

	DEPENDENT_CODE_START( useLayer5 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_5 )
		TSnormal = normalize( _input.normal_and_binormal_z.xyz );
#endif
	DEPENDENT_CODE_ELSE( useLayer5 )
#if defined( _3DSMAX_ ) || !defined( USE_LAYER_5 )
		TSnormal = normalize( _input.normal );
#endif
	DEPENDENT_CODE_END( useLayer5 )

#else
	// Create tangent space vectors
	float3 TSnormal;
	float3 tangent;
	float3 binormal;
	float3 eye = normalize( _input.eye.xyz );

	DEPENDENT_CODE_START( useLayer5 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_5 )
		TSnormal = normalize( _input.normal_and_binormal_z.xyz );
		tangent = normalize( _input.tangent );
		binormal = normalize( float3( _input.texCoord4_and_binormal_xy.zw, _input.normal_and_binormal_z.w ) );
#endif
	DEPENDENT_CODE_ELSE( useLayer5 )
#if defined( _3DSMAX_ ) || !defined( USE_LAYER_5 )

#if defined( _3DSMAX_ )
		TSnormal = normalize( _input.normal_and_binormal_z.xyz );
		tangent = normalize( _input.tangent );
		binormal = normalize( float3( _input.texCoord4_and_binormal_xy.zw, _input.normal_and_binormal_z.w ) );
#else
		TSnormal = normalize( _input.normal );
		tangent = normalize( _input.tangent );
		binormal = normalize( _input.binormal );
#endif

#endif
	DEPENDENT_CODE_END( useLayer5 )

	// Perturb the tangent space normal by the normal map
	float3 normalFromMap = ( layer2NormalMapColour.rgb * 2.0f ) - 1.0f;
	float3 layer2Normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );

#endif

	float4 diffuseTexColour;
	float4 specularTexColour;
	float3 normal;
	float minSpecPowerValue;
	float maxSpecPowerValue;
	float globalSpecularFactorValue = layer2GlobalSpecularFactor;

	// Interpolate the diffuse colour from the layers, specular tex colour, lighting factors and normal from the layers
	diffuseTexColour = layer1DiffuseTexColour * layer2DiffuseTexColour;
	specularTexColour = layer2SpecularTexColour;
#if defined( DISABLE_NORMAL_MAPPING )
		normal = TSnormal;
#else
		normal = layer2Normal;
#endif
	float4 colourLerpTarget = float4( 1.0f, 1.0f, 1.0f, specularTexColour.a );

	DEPENDENT_CODE_START( useLayer3 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_3 )
		diffuseTexColour = lerp( diffuseTexColour, layer3TexColour, layer3TexColour.a );
		specularTexColour = lerp( specularTexColour, colourLerpTarget, layer3TexColour.a );
		globalSpecularFactorValue = lerp( globalSpecularFactorValue, layer3GlobalSpecularFactor, layer3TexColour.a );
#endif
	DEPENDENT_CODE_END( useLayer3 )

	DEPENDENT_CODE_START( useLayer4 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_4 )
		diffuseTexColour = lerp( diffuseTexColour, layer4TexColour, layer4TexColour.a );
		specularTexColour = lerp( specularTexColour, colourLerpTarget, layer4TexColour.a );

#if !defined( DISABLE_NORMAL_MAPPING )

		DEPENDENT_CODE_START( useNormalSmoothing )
#if defined( _3DSMAX_ ) || defined( USE_NORMAL_SMOOTHING )
			normal = lerp( normal, TSnormal, saturate( layer4NormalSmoothingFactor * layer4TexColour.a ) );
#endif
		DEPENDENT_CODE_END( useNormalSmoothing )

#endif

		globalSpecularFactorValue = lerp( globalSpecularFactorValue, layer4GlobalSpecularFactor, layer4TexColour.a );
#endif
	DEPENDENT_CODE_END( useLayer4 )
	
	DEPENDENT_CODE_START( useLayer5 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_5 )
		if ( layer5Additive )
		{
			diffuseTexColour += layer5TexColour;
		}
		else
		{
			diffuseTexColour *= layer5TexColour;
		}
#endif
	DEPENDENT_CODE_END( useLayer5 )

	minSpecPowerValue = layer2MinSpecPower;
	maxSpecPowerValue = layer2MaxSpecPower;
#if !defined( DISABLE_NORMAL_MAPPING )
	normal = normalize( normal );
#endif

	diffuseTexColour *= _input.colour2;

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = layer1DiffuseTexColour.w;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}



//
// Low Detail Shaders
//


// Output structure
struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float4 texCoord01	: TEXCOORD0;												// UV coords for texture channels 0 & 1
#if defined( _3DSMAX_ ) || defined( USE_LAYER_3 ) || defined( USE_LAYER_4 )
	// This vertex stream isn't needed without layer 3 or layer 4
	float4 texCoord23	: TEXCOORD1;												// UV coords for texture channels 2 & 3
#endif
#if defined( _3DSMAX_ ) || defined( USE_LAYER_5 )
	// This vertex stream isn't needed without layer 5
	float2 texCoord4	: TEXCOORD2;												// UV coords for texture channel 4
#endif
};


VSOUTPUT_LD RoadLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.texCoord01.xy = _input.texCoord0;
	_output.texCoord01.zw = _input.texCoord1;

#if defined( _3DSMAX_ ) || defined( USE_LAYER_3 ) || defined( USE_LAYER_4 )
		_output.texCoord23 = float4( 0.0f, 0.0f, 0.0f, 0.0f );
#endif

	DEPENDENT_CODE_START( useLayer3 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_3 )
		_output.texCoord23.xy = _input.texCoord2;
#endif
	DEPENDENT_CODE_END( useLayer3 )

	DEPENDENT_CODE_START( useLayer4 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_4 )
		_output.texCoord23.zw = _input.texCoord3;
#endif
	DEPENDENT_CODE_END( useLayer4 )

	DEPENDENT_CODE_START( useLayer5 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_5 )
		_output.texCoord4 = _input.texCoord4;
#endif
	DEPENDENT_CODE_END( useLayer5 )

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	float4	diffuseTexColour;
	diffuseTexColour = _input.colour;

	// get normal in world space and do lighting
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	
	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	_output.colour = _input.colour;
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT RoadLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 layer1DiffuseTexColour = tex2D( layer1DiffuseMap, _input.texCoord01.xy );
	float4 layer2DiffuseTexColour = tex2D( layer2DiffuseMap, _input.texCoord01.zw );

	float4 diffuseTexColour;

	// Interpolate the diffuse colour from the layers
	diffuseTexColour = layer1DiffuseTexColour * layer2DiffuseTexColour;

	DEPENDENT_CODE_START( useLayer3 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_3 )
		float4 layer3DiffuseTexColour = tex2D( layer3DiffuseMap, _input.texCoord23.xy );
		diffuseTexColour = lerp( diffuseTexColour, layer3DiffuseTexColour, layer3DiffuseTexColour.a );
#endif
	DEPENDENT_CODE_END( useLayer3 )

	DEPENDENT_CODE_START( useLayer4 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_4 )
		float4 layer4DiffuseTexColour = tex2D( layer4DiffuseMap, _input.texCoord23.zw );
		diffuseTexColour = lerp( diffuseTexColour, layer4DiffuseTexColour, layer4DiffuseTexColour.a );
#endif
	DEPENDENT_CODE_END( useLayer4 )

	DEPENDENT_CODE_START( useLayer5 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_5 )
		float4 layer5DiffuseTexColour = tex2D( layer5DiffuseMap, _input.texCoord4 );
		if ( layer5Additive )
		{
			diffuseTexColour += layer5DiffuseTexColour;
		}
		else
		{
			diffuseTexColour *= layer5DiffuseTexColour;
		}
#endif
	DEPENDENT_CODE_END( useLayer5 )

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	accumulatedColour.w = 1.0f;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//

technique road_multilayer
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "road_multilayer";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "road_multilayer_lowdetail";
	int    lowDetailDeferredID		= 0;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx RoadVertexShader();
		PixelShader = compile sce_fp_rsx RoadFragmentShader();
#else		
		VertexShader = compile vs_3_0 RoadVertexShader();
		PixelShader = compile ps_3_0 RoadFragmentShader();
#endif		
	}
}

technique road_multilayer_lowdetail
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "road_multilayer_lowdetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx RoadLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx RoadLowDetailFragmentShader();
#else		
		VertexShader = compile vs_3_0 RoadLowDetailVertexShader();
		PixelShader = compile ps_3_0 RoadLowDetailFragmentShader();
#endif		
	}
}
